"use client";

import { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { sendEmail } from "@/service/ContactService";
import Swal from "sweetalert2";
import { SUPPORT_PHONE } from "@/util/constant";

export default function ContactSalesModal({ plan }) {
    const [open, setOpen] = useState(false);
    const [formData, setFormData] = useState({
        name: "",
        email: "",
        phone: "",
        description: "",
    });
    const [loading, setLoading] = useState(false);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        try {
            const response = await sendEmail(formData);
            if (response.statusCode === 200) {
                Swal.fire({
                    icon: "success",
                    title: response.message || "Email sent successfully to sales team.",
                    toast: true,
                    position: "top-end",
                    showConfirmButton: false,
                    timer: 2000,
                    timerProgressBar: true,
                });
            }
        } catch (error) {
            Swal.fire({
                icon: "error",
                title: "Failed to Send Email",
                text: error.message || "Something went wrong!",
            });
        } finally {
            setLoading(false);
            setOpen(false);
            resetFormData();
        }
    };

    const resetFormData = () => {
    setFormData({
        name: "",
        email: "",
        phone: "",
        description: "",
    });
    };

    return (
        <>
            <Button
                onClick={() => setOpen(true)}
                className={`w-full ${plan.name === "Business" ? "bg-chart-5 hover:bg-chart-5/80" : ""
                    }`}
            >
                Contact Sales
            </Button>

            <Dialog open={open}
                    onOpenChange={(isOpen) => {
                        setOpen(isOpen);
                        if (!isOpen) {
                            // reset form when modal closes
                            setFormData({
                                name: "",
                                email: "",
                                phone: "",
                                description: "",
                            });
                        }
                    }}
                >
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Contact Sales ({plan.name})</DialogTitle>
                    </DialogHeader>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        <Input
                            placeholder="Your Name (required)"
                            value={formData.name}
                            onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                            required
                        />
                        <Input
                            type="email"
                            placeholder="Email"
                            value={formData.email}
                            onChange={(e) => setFormData({ ...formData, email: e.target.value })}                            
                        />
                        <Input
                            type="tel"
                            placeholder="Phone (required)"
                            value={formData.phone}
                            onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                            required
                        />
                        <Textarea
                            placeholder="Tell us about your needs..."
                            value={formData.description}
                            onChange={(e) =>
                                setFormData({ ...formData, description: e.target.value })
                            }
                        />

                        {/* Response Time Notice */}
                        <div className="mt-4 p-4 bg-blue-50 rounded-lg">
                        <p className="text-sm">
                            <strong>Response Time:</strong> We typically respond to all inquiries within 24 hours during business days. <br/>
                            For urgent matters, please call us directly at {SUPPORT_PHONE}.
                        </p>
                        </div>

                        <Button type="submit"
                            disabled={loading} // ✅ disable while sending
                            className={`px-4 py-2 rounded text-white w-full ${loading ? "bg-gray-400 cursor-not-allowed" : "bg-primary hover:bg-primary/80"
                                }`}>
                            {loading ? "Sending..." : "Send"}
                        </Button>
                    </form>
                </DialogContent>
            </Dialog>
        </>
    );
}
