"use client";

import React, { useEffect, useState } from "react";
import { Truck, Caravan, DollarSign } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { getCurrentUser, getMe } from "@/service/AuthService";
import { useRouter } from "next/navigation";

export default function ListingForm({ meta, onSubmit }) {
  const [form, setForm] = useState({
    year: "",
    condition: "",
    type: "",
    categoryId: "",
    makerId: "",
    modelId: "",
    trimId: "",
    truckSpecDTO: {
      truckClassId: "",
      fuelTypeId: "",
      mileage: "",
      vinNumber: "",
      horsePower: "",
      stockNumber: "",
      engineManufacturer: "",
      driverSide: ""
    },
    trailerSpecDTO: {
      trailerType: "",
      length: "",
      vinNumber: "",
      stockNumber: "",
      width: "",
      color: "",
      height: "",
      composition: "",
      numberOfAxles: "",
      axleConfiguration: "",
      hitch: "",
      floorType: "",
      numberOfSideDoors: "",
      rearOpening: "",
      trailerWeightLbs: "",
      trailerCapacityLbs: ""
    },
    title: "",
    address: "",
    city: "",
    state: "",
    country: "",
    description: "",
    price: "",
    userPlanId: ""
  });

  const [errors, setErrors] = useState({});
  const [userPlans, setUserPlans] = useState([]);
  const router = useRouter();

  useEffect(() => {
    const savedForm = localStorage.getItem("pendingListing");
    if (savedForm) {
      setForm(JSON.parse(savedForm));
    }
  }, []);

  useEffect(() => {
    const fetchProfile = async () => {
    const user = getCurrentUser();
    if (user) {
      try {
        const data = await getMe();
        setUserPlans(data.userPlans);
      } catch (error) {
        console.error(error);
      }
    }};
    fetchProfile();
  }, []);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setForm((prev) => ({ ...prev, [name]: value }));
    setErrors((prev) => ({ ...prev, [name]: undefined }));
  };

  const handleSpecChange = (e, spec) => {
    const { name, value } = e.target;
    setForm((prev) => ({
      ...prev,
      [spec]: {
        ...prev[spec],
        [name]: value
      }
    }));
    setErrors((prev) => ({ ...prev, [name]: undefined }));
  };

  const validateStep = () => {
    const newErrors = {};
    const currentYear = new Date().getFullYear();
    const yearNumber = parseInt(form.year, 10);

    // Step 1
    if (!form.year) {
      newErrors.year = "Required";
    } else if (isNaN(yearNumber) || yearNumber < 1980 || yearNumber > currentYear) {
      newErrors.year = `Year must be between 1980 and ${currentYear}`;
    }
    if (!form.type) newErrors.type = "Required";
    if (!form.condition) newErrors.condition = "Required";
    if (!form.categoryId) newErrors.categoryId = "Required";
    if (!form.makerId) newErrors.makerId = "Required";

    // Step 2
    if (!form.modelId) newErrors.modelId = "Required";
    if (!form.trimId) newErrors.trimId = "Required";

    if (form.type === "TRUCK") {
      const t = form.truckSpecDTO;
      if (!t.truckClassId) newErrors.truckClassId = "Required";
      if (!t.fuelTypeId) newErrors.fuelTypeId = "Required";
      if (!t.mileage) {
        newErrors.mileage = "Required";
      } else if (isNaN(Number(t.mileage)) || Number(t.mileage) <= 0 || Number(t.mileage) > 999999) {
        newErrors.mileage = "Mileage must be a number between 1 and 999999";
      }
      if (!t.vinNumber) newErrors.vinNumber = "Required";
      if (!t.stockNumber) newErrors.stockNumber = "Required";
      if (!t.horsePower) newErrors.horsePower = "Required";
      if (!t.engineManufacturer) newErrors.engineManufacturer = "Required";
      if (!t.driverSide) newErrors.driverSide = "Required";

    } else if (form.type === "TRAILER") {
      const t = form.trailerSpecDTO;
      if (!t.trailerType) newErrors.trailerType = "Required";
      if (!t.length) newErrors.length = "Required";

      if (!t.width) newErrors.width = "Required";
      if (!t.vinNumber) newErrors.vinNumber = "Required";
      if (!t.stockNumber) newErrors.stockNumber = "Required";
      if (!t.color) newErrors.color = "Required";
      if (!t.hitch) newErrors.hitch = "Required";
      if (!t.height) newErrors.height = "Required";
      if (!t.numberOfAxles) newErrors.numberOfAxles = "Required";
      if (!t.numberOfSideDoors) newErrors.numberOfSideDoors = "Required";
      if (!t.rearOpening) newErrors.rearOpening = "Required";
      if (!t.axleConfiguration) newErrors.axleConfiguration = "Required";
      if (!t.floorType) newErrors.floorType = "Required";
      if (!t.trailerWeightLbs) newErrors.trailerWeightLbs = "Required";
      if (!t.trailerCapacityLbs) newErrors.trailerCapacityLbs = "Required";
      if (!t.composition) newErrors.composition = "Required";
    }

    // Step 3
    if (!form.title) {
      newErrors.title = "Required";
    } else if (form.title.length > 100) {
      newErrors.title = "Title cannot exceed 100 characters";
    }
    if (!form.address) newErrors.address = "Required";
    if (!form.city) newErrors.city = "Required";
    if (!form.state) newErrors.state = "Required";
    if (!form.country) newErrors.country = "Required";
    if (!form.description) newErrors.description = "Required";

    // Step 4
    if (!form.price) {
      newErrors.price = "Required";
    } else if (isNaN(Number(form.price)) || Number(form.price) <= 0 || Number(form.price) > 999999) {
      newErrors.price = "Price must be a valid positive number";
    }

    if (!form.userPlanId) newErrors.userPlanId = "Required";

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleFinalSubmit = (e) => {
    e.preventDefault();

    const user = getCurrentUser();
    if (!user) {
      // save draft before login
    localStorage.setItem("pendingListing", JSON.stringify(form));
    router.push("/register");
      return;
    }

    if (validateStep()) {
      onSubmit(form);
    }
  };

  const filteredModels = meta.models?.filter(
    (m) => m.type === form.type && m.makerId === form.makerId
  );
  const filteredTrims = meta.trims?.filter((t) => t.modelId === form.modelId);
  const filteredCategories = meta.categories?.filter((c) => {
    return !form.type || form.type === "" || c.type === form.type;
  });
  return (
    <form onSubmit={handleFinalSubmit} className="max-w-4xl mx-auto p-6 border rounded-lg shadow space-y-6">
      <h2 className="flex items-center text-2xl font-semibold leading-none tracking-tight"><Truck className="mr-2" />Vehicle Information</h2>

        <div className="flex gap-4">
          <Button
            type="button"
            variant={form.type === "TRUCK" ? "default" : "outline"}
            onClick={() => handleChange({ target: { name: "type", value: "TRUCK" } })}
            className={`flex items-center gap-2 ${errors.type && "border-red-500"}`}
          >
            <Truck size={20} /> Truck
          </Button>
          <Button
            type="button"
            variant={form.type === "TRAILER" ? "default" : "outline"}
            onClick={() => handleChange({ target: { name: "type", value: "TRAILER" } })}
            className={`flex items-center gap-2 ${errors.type && "border-red-500"}`}
          >
            <Caravan size={20} /> Trailer
          </Button>
        </div>
      {/* Step 1: Basic Info */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <input
          name="year"
          value={form.year}
          onChange={handleChange}
          placeholder="Make Year"
          className={`border p-3 rounded w-full ${errors.year && "border-red-500"}`}
        />
        <input
          name="condition"
          value={form.condition}
          onChange={handleChange}
          placeholder="Condition.(New or Used)"
          className={`border p-3 rounded w-full ${errors.condition && "border-red-500"}`}
        />
        <select
          name="categoryId"
          value={form.categoryId}
          onChange={handleChange}
          className={`border p-3 rounded w-full ${errors.categoryId && "border-red-500"}`}
        >
          <option value="">Select Category</option>
          {filteredCategories?.map(c => (
            <option key={c.id} value={c.id}>{c.name}</option>
          ))}
        </select>
        <select
          name="makerId"
          value={form.makerId}
          onChange={handleChange}
          className={`border p-3 rounded w-full ${errors.makerId && "border-red-500"}`}
        >
          <option value="">Select Maker</option>
          {meta.makers?.map(m => (
            <option key={m.id} value={m.id}>{m.name}</option>
          ))}
        </select>
      </div>

      {/* Step 2: Specs */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <select
          name="modelId"
          value={form.modelId}
          onChange={handleChange}
          className={`border p-3 rounded w-full ${errors.modelId && "border-red-500"}`}
        >
          <option value="">Select Model</option>
          {filteredModels?.map(m => (
            <option key={m.id} value={m.id}>{m.name}</option>
          ))}
        </select>
        <select
          name="trimId"
          value={form.trimId}
          onChange={handleChange}
          className={`border p-3 rounded w-full ${errors.trimId && "border-red-500"}`}
        >
          <option value="">Select Trim</option>
          {filteredTrims?.map(t => (
            <option key={t.id} value={t.id}>{t.name}</option>
          ))}
        </select>

        {form.type === "TRUCK" && (
          <>
            <select
              name="truckClassId"
              value={form.truckSpecDTO.truckClassId}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              className={`border p-3 rounded w-full ${errors.truckClassId && "border-red-500"}`}
            >
              <option value="">Truck Class</option>
              {meta.truckClasses?.map(tc => (
                <option key={tc.id} value={tc.id}>{tc.name} - {tc.gvwrRange}</option>
              ))}
            </select>
            <select
              name="fuelTypeId"
              value={form.truckSpecDTO.fuelTypeId}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              className={`border p-3 rounded w-full ${errors.fuelTypeId && "border-red-500"}`}
            >
              <option value="">Fuel Type</option>
              {meta.fuelTypes?.map(ft => (
                <option key={ft.id} value={ft.id}>{ft.name}</option>
              ))}
            </select>
            <input
              name="mileage"
              value={form.truckSpecDTO.mileage}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="Mileage"
              className={`border p-3 rounded w-full ${errors.mileage && "border-red-500"}`}
            />
            <input
              name="vinNumber"
              value={form.truckSpecDTO.vinNumber}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="VIN Number"
              className={`border p-3 rounded w-full ${errors.vinNumber && "border-red-500"}`}
            />
            <input
              name="stockNumber"
              value={form.truckSpecDTO.stockNumber}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="Stock Number"
              className={`border p-3 rounded w-full ${errors.stockNumber && "border-red-500"}`}
            />
            <input
              name="horsePower"
              value={form.truckSpecDTO.horsePower}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="Horse Power"
              className={`border p-3 rounded w-full ${errors.horsePower && "border-red-500"}`}
            />
            <input
              name="engineManufacturer"
              value={form.truckSpecDTO.engineManufacturer}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="Engine Manufacturer"
              className={`border p-3 rounded w-full ${errors.engineManufacturer && "border-red-500"}`}
            />
            <input
              name="driverSide"
              value={form.truckSpecDTO.driverSide}
              onChange={(e) => handleSpecChange(e, "truckSpecDTO")}
              placeholder="Driver Side"
              className={`border p-3 rounded w-full ${errors.driverSide && "border-red-500"}`}
            />
          </>
        )}

        {form.type === "TRAILER" && (
          <>
            <input
              name="trailerType"
              value={form.trailerSpecDTO.trailerType}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Trailer Type"
              className={`border p-3 rounded w-full ${errors.trailerType && "border-red-500"}`}
            />
            <input
              name="length"
              value={form.trailerSpecDTO.length}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Length in meters"
              className={`border p-3 rounded w-full ${errors.length && "border-red-500"}`}
            />
            <input
              name="width"
              value={form.trailerSpecDTO.width}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Width in meters"
              className={`border p-3 rounded w-full ${errors.width && "border-red-500"}`}
            />
            <input
              name="vinNumber"
              value={form.trailerSpecDTO.vinNumber}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="VIN Number"
              className={`border p-3 rounded w-full ${errors.vinNumber && "border-red-500"}`}
            />
            <input
              name="stockNumber"
              value={form.trailerSpecDTO.stockNumber}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Stick Number"
              className={`border p-3 rounded w-full ${errors.stockNumber && "border-red-500"}`}
            />
            <input
              name="color"
              value={form.trailerSpecDTO.color}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Color"
              className={`border p-3 rounded w-full ${errors.color && "border-red-500"}`}
            />
            <input
              name="height"
              value={form.trailerSpecDTO.height}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Height"
              className={`border p-3 rounded w-full ${errors.height && "border-red-500"}`}
            />
            <input
              name="composition"
              value={form.trailerSpecDTO.composition}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Composition"
              className={`border p-3 rounded w-full ${errors.composition && "border-red-500"}`}
            />
            <input
              name="numberOfAxles"
              value={form.trailerSpecDTO.numberOfAxles}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Number Of Axles"
              className={`border p-3 rounded w-full ${errors.numberOfAxles && "border-red-500"}`}
            />
            <input
              name="axleConfiguration"
              value={form.trailerSpecDTO.axleConfiguration}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Axle Configuration"
              className={`border p-3 rounded w-full ${errors.axleConfiguration && "border-red-500"}`}
            />
            <input
              name="hitch"
              value={form.trailerSpecDTO.hitch}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Hitch"
              className={`border p-3 rounded w-full ${errors.hitch && "border-red-500"}`}
            />
            <input
              name="floorType"
              value={form.trailerSpecDTO.floorType}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Floor Type"
              className={`border p-3 rounded w-full ${errors.floorType && "border-red-500"}`}
            />
            <input
              name="numberOfSideDoors"
              value={form.trailerSpecDTO.numberOfSideDoors}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="No. of Side Doors"
              className={`border p-3 rounded w-full ${errors.numberOfSideDoors && "border-red-500"}`}
            />
            <input
              name="rearOpening"
              value={form.trailerSpecDTO.rearOpening}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Rear Opening (Yes or NO)"
              className={`border p-3 rounded w-full ${errors.rearOpening && "border-red-500"}`}
            />
            <input
              name="trailerWeightLbs"
              value={form.trailerSpecDTO.trailerWeightLbs}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Trailer weight (Lbs)"
              className={`border p-3 rounded w-full ${errors.trailerWeightLbs && "border-red-500"}`}
            />
            <input
              name="trailerCapacityLbs"
              value={form.trailerSpecDTO.trailerCapacityLbs}
              onChange={(e) => handleSpecChange(e, "trailerSpecDTO")}
              placeholder="Trailer capacity (Lbs)"
              className={`border p-3 rounded w-full ${errors.trailerCapacityLbs && "border-red-500"}`}
            />
          </>
        )}
      </div>

      {/* Step 3: Location & Description */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <input name="title" value={form.title} onChange={handleChange} placeholder="Title"
          className={`border p-3 rounded w-full ${errors.title && "border-red-500"}`} />
        <input name="address" value={form.address} onChange={handleChange} placeholder="Address"
          className={`border p-3 rounded w-full ${errors.address && "border-red-500"}`} />
      </div>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <input name="city" value={form.city} onChange={handleChange} placeholder="City"
          className={`border p-3 rounded w-full ${errors.city && "border-red-500"}`} />
        <input name="state" value={form.state} onChange={handleChange} placeholder="State"
          className={`border p-3 rounded w-full ${errors.state && "border-red-500"}`} />
        <input name="country" value={form.country} onChange={handleChange} placeholder="Country"
          className={`border p-3 rounded w-full ${errors.country && "border-red-500"}`} />
      </div>
      <Textarea
        name="description"
        value={form.description}
        onChange={handleChange}
        placeholder="Vehicle Description"
        rows={4}
        className={`border p-3 rounded w-full ${errors.description && "border-red-500"}`}
      />

      {/* Step 4: Price */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div className="flex items-center gap-2">
          <DollarSign />
          <input
            name="price"
            value={form.price}
            onChange={handleChange}
            placeholder="Price"
            className={`border p-3 rounded w-full ${errors.price && "border-red-500"}`}
          />
        </div>

        <select
          name="userPlanId"
          value={form.userPlanId}
          onChange={handleChange}
          className={`border p-3 rounded w-full ${errors.userPlanId && "border-red-500"}`}
        >
          <option value="">Select Plan</option>
          {userPlans?.map(up => (
            <option key={up.id} value={up.id}>{up.planName}</option>
          ))}
        </select>
      </div>
      
      {/* Terms & Conditions checkbox with label */}
      <div className="flex items-center gap-2 md:mt-0">
        <input
          id="agree"
          type="checkbox"
          className="h-4 w-4 border-gray-300 rounded"
          onChange={() => { /* dummy checkbox for now */ }}
        />
        <label htmlFor="agree" className="text-sm text-gray-700">
          I agree to the <span className="underline cursor-pointer">Terms & Conditions</span>
        </label>
      </div>
      
      <div className="flex justify-center mt-6">
        <Button type="submit" className="h-11">
          <div className="flex items-center justify-center gap-2 px-8">
            <Truck className="h-5 w-5" />
            List My Truck
          </div>
        </Button>
      </div>

    </form>
  );
}
