// lib/services/authService.js
import { API_BASE_URL } from "@/util/BaseUrl";

const AUTH_URL = `${API_BASE_URL}/api/auth/user`;
const LOGOUT_URL = `${API_BASE_URL}/logout`;

const STORAGE_KEY = "user"; // stores { token, username, role }

// Login — calls backend, stores entire response (token + username + role)
export const login = async (username, password) => {
  const res = await fetch(`${AUTH_URL}/login`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ username, password }),
  });

  if (!res.ok) {
    const data = await res.json();    
    throw new Error(data.message || "Failed to login.");
  }

  const data = await res.json();  
  localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
  return data;
};

// Get logged-in user from storage (token, name, role)
export const getCurrentUser = () => {
  const stored = localStorage.getItem(STORAGE_KEY);
  return stored ? JSON.parse(stored) : null;
};


// Get fresh user details by verifying token on backend
export const getMe = async () => {
  const user = getCurrentUser();
  if (!user?.token) throw new Error("No token found");

  const res = await fetch(`${AUTH_URL}/me`, {
    headers: {
      Authorization: `Bearer ${user.token}`,
    },
  });

  if (!res.ok) {
    throw new Error("Failed to get user details");
  }

  return res.json();
};

// 🚀 Send OTP
export const sendOtp = async (phoneNumber) => {
  const res = await fetch(`${AUTH_URL}/otp-send?phoneNumber=${encodeURIComponent(phoneNumber)}`, {
    method: "POST",
  });

  if (!res.ok) {
    const data = await res.json();
    throw new Error(data.message || "Failed to send OTP.");
  }

  return res.json(); // backend sends CustomResponse
};

// 🚀 Verify OTP
export const verifyOtp = async (phoneNumber, otpCode) => {
  const res = await fetch(
    `${AUTH_URL}/otp-verify?phoneNumber=${encodeURIComponent(phoneNumber)}&otpCode=${encodeURIComponent(otpCode)}`,
    { method: "POST" }
  );

  if (!res.ok) {
    const data = await res.json();
    throw new Error(data.message || "Failed to verify OTP.");
  }

  return res.json(); // backend sends CustomResponse
};

// Logout: hit backend + clear local
export const logout = async () => {
  const user = getCurrentUser();
  if (user?.token) {
    await fetch(LOGOUT_URL, {
      method: "POST",
      headers: {
        Authorization: `Bearer ${user.token}`,
      },
    });
  }
  localStorage.removeItem(STORAGE_KEY);
};

const AuthService = {
  login,
  getCurrentUser,
  getMe,
  sendOtp,
  verifyOtp,
  logout,
};

export default AuthService;
